/*
 * FMS routines for managing FMAs probing the fabric for verification
 * and discovery.
 */
#include <sys/types.h>
#include <netinet/in.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fma_comm.h"
#include "lf_channel.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_fma.h"
#include "fms_settings.h"
#include "fms_notify.h"

void dumpit(struct lf_fabric *fp);

static void
fms_add_host_probe(
  struct fms_fma_desc *adp,
  struct fms_link *linkp,
  int link_topo_index)
{
  struct fms_probe *fpp;

  LF_CALLOC(fpp, struct fms_probe, 1);

  /* Fill in and link the probe definition */
  fpp->link_index = link_topo_index;
  fms_link_probe_definition(fpp, adp, linkp);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Clear all probes in the fabric, and return a count of xbar-xbar links
 */
void
fms_destroy_all_probes(
  struct lf_fabric *fp)
{
  int h;

  /*
   * loop through all the hosts in the fabric and release all probe
   * assignments.
   */
  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    /* If there is an FMA for this host, destroy its probes */
    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) {
      fms_destroy_probes_for_host(adp);
    }
  }
}

/*
 * Count the total number of xbar-xbar links in the fabric
 */
int
fms_count_xbar_links(
  struct lf_fabric *fp)
{
  int x;
  int nl;

  /*
   * loop through all xbars and count links to other xbars
   */
  nl = 0;
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    int p;

    xp = fp->xbars[x];
    for (p=0; p<xp->num_ports; ++p) {
      struct lf_xbar *oxp;

      oxp = LF_XBAR(xp->topo_ports[p]);

      /* visit each link only once */
      if (oxp != NULL
	  && oxp->ln_type == LF_NODE_XBAR
	  && oxp >= xp
	  && (oxp != xp || xp->topo_rports[p] >= p)) {
	++nl;
      }
    }
  }
  
  return nl;
}

/*
 * Assign link probes such that each link in the fabric is tested,
 * spreading the work across the hosts.  Try to get each link assigned to
 * FMS_HOST_PROBES_PER_LINK unique hosts, but never assign a single host to
 * the same link twice.
 */
void
fms_make_probe_assignments(
  struct lf_fabric *fp,
  struct lf_host **hosts,
  int num_hosts)
{
  int hosts_per_link;
  int live_hosts;
  int num_probes;
  int num_links;
  int bf_serial;
  int h;

  fms_notify(FMS_EVENT_DEBUG, "Making probe assignments");

  /* Only use hosts with a valid FMA associated */
  live_hosts = 0;
  for (h=0; h<num_hosts; ++h) {
    if (FMS_HOST(hosts[h])->adp != NULL) {
      ++live_hosts;
    }
  }

  /* target number of hosts for each link */
  hosts_per_link = FMS_HOST_PROBES_PER_LINK;
  if (hosts_per_link > live_hosts) hosts_per_link = live_hosts;

  /* clear all current probes and count the links that are up */
  fms_destroy_all_probes(fp);
  num_links = fms_count_xbar_links(fp);

  /* this is the number of xbar-xbar probes we expect to create */
  num_probes = num_links * hosts_per_link;

  /*
   * Keep looping over hosts until all probes are created
   * XXX time vs. space - do parallel bfs from each interface?
   */

  /* clear all serial numbers to indicate not visited */
  bf_serial = 0;
  fms_clear_bf_serial(fp);

  while (num_probes > 0) {
    int old_num_probes;

    /*
     * for each interface on each NIC on each host, find a new probe to do
     */
    old_num_probes = num_probes;

    for (h=0; h<num_hosts; ++h) {
      struct lf_host *hp;
      struct fms_fma_desc *adp;
      int n;

      hp = hosts[h];

      /* skip hosts with no FMA */
      adp = FMS_HOST(hp)->adp;
      if (adp == NULL) continue;

      for (n=0; n<hp->num_nics; ++n) {
	struct lf_nic *nicp;
	int p;

	nicp = hp->nics[n];

	for (p=0; p<nicp->num_ports; ++p) {
	  struct lf_xbar *search_list;

	  /* don't explore out downed NIC links */
	  if (nicp->link_state[p] != LF_LINK_STATE_UP) continue;

	  /* find direct connection and make sure it is an xbar */
	  search_list = LF_XBAR(nicp->topo_ports[p]);
	  if (search_list == NULL || search_list->ln_type != LF_NODE_XBAR) {
	    continue;
	  }

	  /* start a new breadth-first search */
	  ++bf_serial;
	  FMS_XBAR(search_list)->bf_serial = bf_serial;
	  FMS_XBAR(search_list)->bf_next = NULL;

	  while (search_list != NULL) {
	    struct lf_xbar *xp;

	    /* process this list looking for a link to probe */
	    xp = search_list;
	    search_list = NULL;

	    while (xp != NULL) {
	      int p;

	      for (p=0; p<xp->num_ports; ++p) {
		struct lf_xbar *oxp;

		/* process this neighbor if it is a crossbar of higher or
		 * equal rank to our own
		 */
		oxp = LF_XBAR(xp->topo_ports[p]);
		if (oxp != NULL
		    && oxp->ln_type == LF_NODE_XBAR
		    && oxp->clos_level >= xp->clos_level) {
		  struct fms_link *xlinkp;

		  /* pointer for this link */
		  xlinkp = FMS_XBAR(xp)->ports[p]->link;

		  /* probe this link if state is UP or DOWN (not MAINT or
		   * FLAKY) and if there is a slot open
		   */
		  if ((xp->link_state[p] == LF_LINK_STATE_UP ||
		       xp->link_state[p] == LF_LINK_STATE_DOWN)
		      && xlinkp->num_probers < hosts_per_link) {
		    struct fms_probe *fpp;

		    /* make sure this host not already probing this link */
		    fpp = xlinkp->probe_anchor->link_next;
		    while (fpp != xlinkp->probe_anchor) {
		      if (fpp->adp->fabric_host == hp) {
			goto next_probe_candidate;
		      }
		      fpp = fpp->link_next;
		    }

		    /* if we made it here, assign this probe */
		    fms_add_host_probe(adp, xlinkp, xp->link_topo_index[p]);
		    --num_probes;
		    goto try_next_nic_port;
		  }

		 next_probe_candidate:

		  /* if not yet visited, add oxp to next level search */
		  if (FMS_XBAR(oxp)->bf_serial != bf_serial) {
		    FMS_XBAR(oxp)->bf_serial = bf_serial;
		    FMS_XBAR(oxp)->bf_next = search_list;
		    search_list = oxp;
		  }
		}
	      }

	      /* next at this leven of breadth-first search */
	      xp = FMS_XBAR(xp)->bf_next;
	    }
	  }
	 try_next_nic_port:
	  search_list = NULL;		/* no-op for the label :-P */
	}
      }
    }

    /* make sure each loop makes progress */
    if (num_probes == old_num_probes) {
      fms_notify(FMS_EVENT_DEBUG, "no progress with %d probes to go",
	           num_probes);
      break;
    }
  }

  /* XXX - send all probe assignments to FMAs */
  if (F.debug) dumpit(fp);
  fms_fma_clear_all_verify_probes(fp);
  fms_fma_define_all_verify_probes(fp);
}

/*
 * Tell all FMAs to clear their verifies.
 */
void
fms_fma_clear_all_verify_probes(
  struct lf_fabric *fp)

{
  int h;

  fms_notify(FMS_EVENT_DEBUG, "Clearing verify probes");

  /* Clear verify probes on each FMA */
  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) {
      int rc;

      /* clear the probes for this host */
      rc = fms_fma_clear_verify_probes(adp);
      if (rc == -1) LF_ERROR(("Error while clearing all verify probes"));
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Send probes definitions all FMAs
 */
void
fms_fma_define_all_verify_probes(
  struct lf_fabric *fp)
{
  int h;

  /* Define verify probes on each FMA */
  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) {
      int rc;

      /* clear the probes for this host */
      rc = fms_fma_send_verify_probes(adp);
      if (rc == -1) LF_ERROR(("Error while clearing all verify probes"));
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Clear the probes on an FMA
 */
int
fms_fma_clear_verify_probes(
  struct fms_fma_desc *adp)
{
  int rc;

  /* send command to clear all verify probes */
  rc = fms_fma_write(adp, LF_FMA_CLEAR_PROBES, NULL, 0);
  if (rc == -1) {
    LF_ERROR(("Error sending clear_probes to %s", adp->hostname));
  }
  return 0;

 except:
  return -1;
}

/*
 * Send all verify probes for an FMA
 */
int
fms_fma_send_verify_probes(
  struct fms_fma_desc *adp)
{
  struct fms_probe *fpp;
  int rc;
  int np;

  /* Send every probe definition for this host */
  np = 0;
  for (fpp = adp->probe_anchor->host_next;
       fpp != adp->probe_anchor;
       fpp = fpp->host_next) {
    rc = fms_fma_define_verify_probe(fpp);
    if (rc == -1) LF_ERROR(("Error defining verify probe to FMA"));
    ++np;
  }

  fms_notify(FMS_EVENT_DEBUG, "Defined %d probes to %s", np, adp->hostname);

  return 0;

 except:
  return -1;
}

/*
 * Send a verify probe assignment to an FMA
 */
int
fms_fma_define_verify_probe(
  struct fms_probe *fpp)
{
  struct lf_fma_define_probe defprobe;
  struct fms_fabric *ffp;
  struct fms_settings *fsp;
  int rc;

  /* get to our local fabric struct and settings*/
  ffp = FMS_FABRIC(F.fabvars->fabric);
  fsp = F.settings;

  /* build the probe message */
  defprobe.topo_index_32 = htonl(fpp->link_index);

  /* send header for probe definition */
  rc = fms_fma_write(fpp->adp, LF_FMA_DEFINE_PROBE,
		     &defprobe, sizeof(defprobe));
  if (rc == -1) {
    LF_ERROR(("Error sending define probe to %s", fpp->adp->hostname));
  }
  return 0;

 except:
  return -1;
}

/*
 * Allocate an empty probe anchor
 */
struct fms_probe *
fms_alloc_probe_anchor()
{
  struct fms_probe *anchor;

  LF_CALLOC(anchor, struct fms_probe, 1);
  anchor->host_next = anchor;
  anchor->host_prev = anchor;
  anchor->link_next = anchor;
  anchor->link_prev = anchor;

  return anchor;

 except:
  return NULL;
}

/*
 * Add a probe definition to both an FMA and a link
 */
void
fms_link_probe_definition(
  struct fms_probe *fpp,
  struct fms_fma_desc *adp,
  struct fms_link *linkp)
{
  /* Add to host list */
  fpp->host_next = adp->probe_anchor->host_next;
  fpp->host_prev = adp->probe_anchor;
  fpp->host_next->host_prev = fpp;
  fpp->host_prev->host_next = fpp;
  fpp->adp = adp;

  /* add to link list */
  fpp->link_next = linkp->probe_anchor->link_next;
  fpp->link_prev = linkp->probe_anchor;
  fpp->link_next->link_prev = fpp;
  fpp->link_prev->link_next = fpp;
  fpp->linkp = linkp;

  /* up the number of probers for this link */
  ++linkp->num_probers;
}

/*
 * Unlink and free this probe struct
 */
void
fms_destroy_probe(
  struct fms_probe *fpp)
{
  /* remove from host list */
  if (fpp->host_next != NULL) {
    fpp->host_next->host_prev = fpp->host_prev;
    fpp->host_prev->host_next = fpp->host_next;
  }

  /* remove from link list */
  if (fpp->link_next != NULL) {
    fpp->link_next->link_prev = fpp->link_prev;
    fpp->link_prev->link_next = fpp->link_next;
    --fpp->linkp->num_probers;
  }

  /* free the probe struct */
  LF_FREE(fpp);
}

/*
 * Destroy all the probes for a host.
 */
void
fms_destroy_probes_for_host(
  struct fms_fma_desc *adp)
{
  struct fms_probe *fpp;

  fms_notify(FMS_EVENT_DEBUG, "destroying probes for %s", adp->hostname);

  /* Just keep destroying top probe until list is empty */
  for (fpp = adp->probe_anchor->host_next;
       fpp != adp->probe_anchor;
       fpp = adp->probe_anchor->host_next) {
    fms_destroy_probe(fpp);
  }
}
